/**
 * WordPress dependencies
 */
import { MediaUploadCheck, BlockControls, MediaUpload } from '@wordpress/block-editor';
import { ToolbarButton, Dropdown, NavigableMenu, MenuItem } from '@wordpress/components';
import { __, sprintf } from '@wordpress/i18n';
import { linkOff, image as imageIcon } from '@wordpress/icons';
/**
 * Internal dependencies
 */
import styles from './style.module.scss';

const VIDEO_POSTER_ALLOWED_MEDIA_TYPES = [ 'image' ];

/**
 * Poster image control react component.
 *
 * @param {object} props                 - Component props.
 * @param {object} props.attributes      - Block attributes.
 * @param {Function} props.setAttributes - Function to set block attributes.
 * @param {string} props.clientId        - Block client ID.
 * @returns {object}                     - React component.
 */
export default function PosterImageBlockControl( { attributes, setAttributes, clientId } ) {
	const { poster } = attributes;
	const onSelectPoster = image => {
		setAttributes( { poster: image.url } );
	};

	const onRemovePoster = () => {
		setAttributes( { poster: '' } );
	};

	const currentImage = () => {
		if ( poster ) {
			return (
				<>
					<span>
						{ __(
							'You are currently overriding the default Poster image. Remove it if you want to use the default image generated by VideoPress.',
							'jetpack-videopress-pkg'
						) }
					</span>
				</>
			);
		}
		return (
			<>
				{ __(
					'No custom Poster image selected. You can upload or select an image from your media library to override the default video image',
					'jetpack-videopress-pkg'
				) }
			</>
		);
	};

	return (
		<BlockControls group="block">
			<Dropdown
				contentClassName={ styles.dropdown_content }
				renderToggle={ ( { isOpen, onToggle } ) => (
					<ToolbarButton
						label={ __( 'Poster image', 'jetpack-videopress-pkg' ) }
						showTooltip
						aria-expanded={ isOpen }
						aria-haspopup="true"
						onClick={ onToggle }
						icon={ imageIcon }
					/>
				) }
				renderContent={ ( { onClose } ) => {
					const videoPosterDescription = `video-block__poster-image-description-${ clientId }`;
					return (
						<>
							<NavigableMenu className="block-editor-media-replace-flow__media-upload-menu">
								<MediaUploadCheck>
									<MediaUpload
										title={ __( 'Select Poster Image', 'jetpack-videopress-pkg' ) }
										onSelect={ image => {
											onSelectPoster( image );
											onClose();
										} }
										allowedTypes={ VIDEO_POSTER_ALLOWED_MEDIA_TYPES }
										render={ ( { open } ) => (
											<MenuItem
												icon={ imageIcon }
												onClick={ open }
												aria-describedby={ videoPosterDescription }
											>
												{ ! poster
													? __( 'Select Poster Image', 'jetpack-videopress-pkg' )
													: __(
															'Replace Poster image',
															'jetpack-videopress-pkg',
															/* dummy arg to avoid bad minification */ 0
													  ) }
												<p id={ videoPosterDescription } hidden>
													{ poster
														? sprintf(
																/* translators: Placeholder is an image URL. */
																__(
																	'The current poster image url is %s',
																	'jetpack-videopress-pkg'
																),
																poster
														  )
														: __(
																'There is no poster image currently selected',
																'jetpack-videopress-pkg'
														  ) }
												</p>
											</MenuItem>
										) }
									/>
								</MediaUploadCheck>

								{ !! poster && (
									<MenuItem
										onClick={ () => {
											onRemovePoster();
											onClose();
										} }
										icon={ linkOff }
									>
										{ __( 'Remove and use default', 'jetpack-videopress-pkg' ) }
									</MenuItem>
								) }
							</NavigableMenu>

							<div className={ styles.current_media }>{ currentImage() }</div>
						</>
					);
				} }
			/>
		</BlockControls>
	);
}
